<?php
/**
 * @package   Gallery
 * @author    Balbooa http://www.balbooa.com/
 * @copyright Copyright @ Balbooa
 * @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
 */

namespace Balbooa\Component\Gallery\Administrator\Helper;

use Joomla\CMS\Factory;
use Joomla\Database\DatabaseInterface;
use DOMDocument;
use DOMElement;
use Joomla\Filesystem\File;

defined('_JEXEC') or die;

class GalleryXmlHelper
{
    public function export(array $pks): string
    {
        $doc = new DOMDocument('1.0');
        $doc->formatOutput = true;

        $root = $doc->createElement('balbooa');
        $doc->appendChild($root);

        $galleries = $this->getItems('#__gallery_galleries', $pks);

        $tagsIds = [];
        $colorsIds = [];

        foreach ($galleries as $gallery) {
            $galleriesElement = $doc->createElement('galleries');
            $root->appendChild($galleriesElement);

            $this->appendJsonElement($doc, $galleriesElement, 'gallery', $gallery);

            $categories = $this->getItems('#__gallery_category', [$gallery->id], 'form_id');
            foreach ($categories as $category) {
                $this->appendJsonElement($doc, $galleriesElement, 'category', $category);
            }

            $images = $this->getItems('#__gallery_items', [$gallery->id], 'form_id');
            foreach ($images as $image) {
                $this->appendJsonElement($doc, $galleriesElement, 'image', $image);
            }

            $tagsIds = array_merge(
                $tagsIds,
                $this->appendMappedData($doc, $galleriesElement, '#__gallery_tags_map', $gallery->id, 'tags_map', 'tag_id')
            );

            $colorsIds = array_merge(
                $colorsIds,
                $this->appendMappedData($doc, $galleriesElement, '#__gallery_colors_map', $gallery->id, 'colors_map', 'color_id')
            );
        }

        $tagsIds = array_unique($tagsIds);
        $colorsIds = array_unique($colorsIds);

        $tags = $this->getItems('#__gallery_tags', $tagsIds);
        $this->appendJsonElement($doc, $root, 'tags', $tags);

        $colors = $this->getItems('#__gallery_colors', $colorsIds);
        $this->appendJsonElement($doc, $root, 'colors', $colors);

        $file = 'galleries-' . time() . '.xml';
        $doc->save($this->path . $file);

        return $file;
    }

    protected string $path = JPATH_ROOT . '/tmp/';

    private function appendJsonElement(DOMDocument $doc, DOMElement $parent, string $name, $data): void
    {
        $element = $doc->createElement($name);
        $json = json_encode($data);
        $element->appendChild($doc->createTextNode($json));
        $parent->appendChild($element);
    }

    private function appendMappedData(DOMDocument $doc, DOMElement $parent, string $table, int $galleryId, string $elementName, string $key): array
    {
        $result = [];
        $items = $this->getItems($table, [$galleryId], 'gallery_id');
        foreach ($items as $item) {
            $this->appendJsonElement($doc, $parent, $elementName, $item);
            $result[] = $item->{$key};
        }

        return $result;
    }

    private function getItems(string $table, $pks, $key = 'id'): array
    {
        if (empty($pks)) {
            return [];
        }

        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->select('*')
            ->from($table)
            ->whereIn($key, $pks);
        $db->setQuery($query);

        return $db->loadObjectList();
    }

    public function download(string $file): void
    {
        $filePath = $this->path . $file;
        if (!file_exists($filePath)) {
            return;
        }

        if (!$this->isXml($filePath)) {
            return;
        }

        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="galleries.xml"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($filePath));

        if (readfile($filePath)) {
            unlink($filePath);
        }
    }

    protected function isXml(string $filePath): bool
    {
        $extension = strtolower(pathinfo($filePath, PATHINFO_EXTENSION));

        return $extension === 'xml';
    }

    protected function loadFile(array $file): \SimpleXMLElement|false
    {
        $path = $this->path . $file['name'];
        File::upload($file['tmp_name'], $path);

        return simplexml_load_file($path);
    }


    public function import(array $file): void
    {
        if (!$this->isXml($file['name'])) {
            return;
        }

        $xml = $this->loadFile($file);
        if (!$xml) {
            return;
        }

        $db = Factory::getContainer()->get(DatabaseInterface::class);

        $tagsMap = $this->importEntities($db, '#__gallery_tags', json_decode((string)$xml->tags));
        $colorsMap = $this->importEntities($db, '#__gallery_colors', json_decode((string)$xml->colors));

        $galleriesMap = (object) [];

        foreach ($xml->galleries as $galleryNode) {
            $gallery = json_decode((string) $galleryNode->gallery);
            $originalGalleryId = $gallery->id;
            unset($gallery->id);
            $db->insertObject('#__gallery_galleries', $gallery, 'id');
            $galleriesMap->{$originalGalleryId} = $gallery->id;

            $categoryMap = $this->importChildren($db, $galleryNode->category, '#__gallery_category', $galleriesMap);

            $imageMap = $this->importChildren($db, $galleryNode->image, '#__gallery_items', $galleriesMap);

            $this->importMap($db, $galleryNode->tags_map, '#__gallery_tags_map', $tagsMap, $imageMap, $galleriesMap, 'tag_id');
            $this->importMap($db, $galleryNode->colors_map, '#__gallery_colors_map', $colorsMap, $imageMap, $galleriesMap, 'color_id');
        }
        unlink($this->path . $file['name']);
    }

    private function importEntities(DatabaseInterface $db, string $table, array $items): object
    {
        $map = (object) [];

        foreach ($items as $item) {
            $id = $item->id;
            unset($item->id);

            $query = $db->getQuery(true)
                ->select('id')
                ->from($table)
                ->where('title' . ' = ' . $db->quote($item->title));
            $db->setQuery($query);
            $existingId = $db->loadResult();

            if ($existingId) {
                $map->{$id} = $existingId;
            } else {
                $db->insertObject($table, $item, 'id');
                $map->{$id} = $item->id;
            }
        }

        return $map;
    }

    private function importChildren(DatabaseInterface $db, $nodes, string $table, object $galleryMap): object
    {
        $map = (object) [];

        foreach ($nodes as $node) {
            $item = json_decode((string) $node);
            $oldId = $item->id;
            unset($item->id);

            $item->form_id = $galleryMap->{$item->form_id};
            $db->insertObject($table, $item, 'id');
            $map->{$oldId} = $item->id;
        }

        return $map;
    }

    private function importMap(
        DatabaseInterface $db,
        \SimpleXMLElement $nodes,
        string $table,
        object $refMap,
        object $imageMap,
        object $galleryMap,
        string $key
    ): void {
        foreach ($nodes as $node) {
            $item = json_decode((string) $node);
            unset($item->id);

            $item->$key = $refMap->{$item->$key} ?? null;

            $item->image_id = $imageMap->{$item->image_id} ?? null;
            $item->gallery_id = $galleryMap->{$item->gallery_id} ?? null;

            $db->insertObject($table, $item, 'id');
        }
    }

}