<?php
/**
* @package   Gridbox
* @author    Balbooa http://www.balbooa.com/
* @copyright Copyright @ Balbooa
* @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
*/

namespace Balbooa\Component\Gridbox\Site\Helper;

use Balbooa\Component\Gridbox\Administrator\Helper\Filesystem\File;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Filter\OutputFilter;

defined('_JEXEC') or die;

abstract class BaseHelper
{
    public static ?BookingHelper $booking = null;
    public static StoreHelper $storeHelper;
    public static $website;
    public static $taxRates;
    public static $store;
    public static $cacheData;
    public static string $side = 'admin';
    public static $path = JPATH_ROOT . '/components/com_gridbox';

    public static function addCacheData(mixed $data, string $key, string $subKey): void
    {
        if (!self::$cacheData) {
            self::$cacheData = (object) [];
        }
        if (!isset(self::$cacheData->{$key})) {
            self::$cacheData->{$key} = new \stdClass();
        }
        self::$cacheData->{$key}->{$subKey} = $data;
    }

    public static function checkInstalledBlog(string $type = '')
    {
        $app_type = empty($type) ? 'blog' : $type;
        if (isset(self::$cacheData->installed->{$app_type})) {
            return self::$cacheData->installed->{$app_type};
        }
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('COUNT(id)')
            ->from('#__gridbox_app')
            ->where('type <> '.$db->quote('system_apps'))
            ->where('type <> '.$db->quote('single'));
        if (!empty($type)) {
            $query->where('type = '.$db->quote($type));
        }
        $db->setQuery($query);
        $count = $db->loadResult();
        self::addCacheData($count, 'installed', $app_type);

        return $count > 0;
    }

    public static function getBooking(): BookingHelper
    {
        if (self::$booking) {
            return self::$booking;
        }
        self::$booking = new BookingHelper();

        return self::$booking;
    }

    public static function getAuthor($id): ?object
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_authors')
            ->where('user_id = '.$id);
        $db->setQuery($query);
        $author = $db->loadObject();

        return $author;
    }

    public static function prepareIntroImage($img)
    {
        if (!is_numeric($img)) {
            return $img;
        }
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_fields_desktop_files')
            ->where('id = '.$img);
        $db->setQuery($query);
        $obj = $db->loadObject();
        if (!isset($obj->filename)) {
            return '';
        }
        
        return 'components/com_gridbox/assets/uploads/app-'.$obj->app_id.'/'.$obj->filename;
    }

    public static function triggerEvent($event, $data = [], $plugin = null)
    {
        if ($plugin) {
            PluginHelper::importPlugin($plugin);
        }
        $dispatcher = Factory::getApplication();
        $dispatcher->triggerEvent($event, $data);
    }

    public static function isExternal($link)
    {
        return (str_contains($link, 'https://') || str_contains($link, 'http://'));
    }

    public static function deleteFolder($dir)
    {
        if (!is_dir($dir)) { 
            return;
        }
        $objects = scandir($dir);
        foreach ($objects as $object) {
            if ($object != "." && $object != "..") {
                if (filetype($dir."/".$object) == "dir") {
                    self::deleteFolder($dir."/".$object);
                } else {
                    unlink($dir."/".$object);
                }
            }
        }
        reset($objects);
        rmdir($dir);
    }

    public static function object_extend($obj1, $obj2) {
        $obj = json_decode(json_encode($obj1));
        $object = json_decode(json_encode($obj2));
        foreach ($object as $key => $value) {
            if (is_object($value)) {
                if (!isset($obj1->{$key})) {
                    $obj->{$key} = $value;
                } else {
                    $obj->{$key} = self::object_extend($obj1->{$key}, $value);
                }
            } else {
                $obj->{$key} = $value;
            }
        }

        return $obj;
    }

    public static function getUserGroups($id = 0)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__usergroups')
            ->order('lft ASC');
        if (!empty($id)) {
            $query->where('id = '.$id);
        }
        $db->setQuery($query);
        if (!empty($id)) {
            $groups = $db->loadObject();
        } else {
            $groups = $db->loadObjectList();
            foreach ($groups as $group) {
                $group->level = self::getUserGroupLevel($group->parent_id);
            }
        }

        return $groups;
    }

    public static function getUserGroupLevel($parent, $level = 0)
    {
        if (empty($parent)) {
            return $level;
        }
        ++$level;
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('parent_id')
            ->from('#__usergroups')
            ->where('id = '.$parent);
        $db->setQuery($query);
        $id = $db->loadResult();
        $level = self::getUserGroupLevel($id, $level);

        return $level;
    }

    public static function removeTmpAttachment($id, $filename, $app = 'comments')
    {
        if (empty($id) || empty($filename)) {
            return;
        }
        $db = Factory::getDbo();
        $dir = JPATH_ROOT.'/components/com_gridbox/assets/uploads/' . $app . '/';
        if (File::exists($dir.$filename)) {
            File::delete($dir.$filename);
        }
        $query = $db->getQuery(true)
            ->delete('#__gridbox_' . $app . '_attachments')
            ->where('id = '.$id);
        $db->setQuery($query)
            ->execute();
    }

    public static function removeTmpReviewsAttachment($id, $filename)
    {
        self::removeTmpAttachment($id, $filename, 'reviews');
    }

    public static function getCommentAttachments($id, $app = 'comments')
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_' . $app . '_attachments')
            ->where('comment_id = '.$id);
        $db->setQuery($query);
        $files = $db->loadObjectList();
        $dir = Uri::root().'components/com_gridbox/assets/uploads/' . $app . '/';
        foreach ($files as $file) {
            $file->link = $dir.$file->filename;
        }

        return $files;
    }

    public static function getReviewAttachments($id)
    {
        return self::getCommentAttachments($id, 'reviews');
    }

    public static function getUserAvatar($email, $key, $author = null)
    {
        $avatar = '';
        if (!empty($email)) {
            $db = Factory::getDbo();
            $query = $db->getQuery(true)
                ->select('a.avatar')
                ->from('#__gridbox_user_avatars AS a')
                ->where('u.email = '.$db->quote($email))
                ->leftJoin('#__users AS u ON u.id = a.user_id');
            $db->setQuery($query);
            $avatar = $db->loadResult();
        }
        $avatar = $author->avatar ?? $avatar;
        if (!empty($avatar) && !self::isExternal($avatar)) {
            $avatar = Uri::root().$avatar;
        } else if (empty($avatar) && self::$website->{$key} == 1 && !empty($email)) {
            $avatar = Uri::root().'components/com_gridbox/assets/images/default-user.png';
            $hash = md5(strtolower(trim($email)));
            $avatar = "https://www.gravatar.com/avatar/".$hash."?d=".$avatar."&s=50";
        } else if (empty($avatar)) {
            $avatar = Uri::root().'components/com_gridbox/assets/images/default-user.png';
        }

        return $avatar;
    }

    public static function getOptions($type)
    {
        $path = JPATH_ROOT.'/components/com_gridbox/libraries/json/'.$type.'.json';
        if (is_file($path)) {
            $json = FileHelper::readFile($path);
        } else {
            $json = '{}';
        }
        
        return json_decode($json);
    }

    public static function getAppCategories($app_id, $id = 0, $level = 0)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select('title, id, app_id')
            ->from('#__gridbox_categories')
            ->where('app_id = '.$app_id)
            ->where('parent = '.$id)
            ->order('order_list ASC');
        $db->setQuery($query);
        $categories = $db->loadObjectList();
        $data = [];
        foreach ($categories as $category) {
            $category->level = $level;
            $data[] = $category;
            $sub = self::getAppCategories($app_id, $category->id, $level + 1);
            $data = array_merge($data, $sub);
        }

        return $data;
    }

    public static function getGridboxLanguage()
    {
        $result = [
            'EDIT_NOT_PERMITTED' => Text::_('JLIB_APPLICATION_ERROR_EDIT_NOT_PERMITTED'),
            'CREATE_NOT_PERMITTED' => Text::_('JERROR_CORE_CREATE_NOT_PERMITTED'),
            'SAVE_SUCCESS' => Text::_('JLIB_APPLICATION_SAVE_SUCCESS'),
            'TITLE' => Text::_('JGLOBAL_TITLE'),
            'ERROR' => Text::_('ERROR'), 'YEAR' => Text::_('JYEAR'),
            'MONTH' => Text::_('JMONTH')
        ];
        $date = Date::getInstance('2023-01-01');
        for ($i = 1; $i <= 12; $i++) {
            if ($i != 1) {
                $date->modify('+1 month');
            }
            $result['SHORT_M'.$i] = $date->format('M');
        }
        if (static::$side == 'admin') {
            $result['MAKE_A_SELECTION'] = Text::_('JLIB_HTML_PLEASE_MAKE_A_SELECTION_FROM_THE_LIST');
        }
        $path = JPATH_ROOT.'/administrator/components/com_gridbox/language/'
            . static::$side .'/en-GB/en-GB.com_gridbox.ini';
        if (File::exists($path)) {
            $contents = FileHelper::readFile($path);
            $contents = str_replace('_QQ_', '"\""', $contents);
            $data = parse_ini_string($contents);
            foreach ($data as $ind => $value) {
                $result[$ind] = Text::_($ind);
            }
        }
        
        $data = 'var gridboxLanguage = '.json_encode($result).';';

        return $data;
    }

    public static function aboutUs()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select("manifest_cache")
            ->from("#__extensions")
            ->where("type=" .$db->quote('component'))
            ->where('element=' .$db->quote('com_gridbox'));
        $db->setQuery($query);
        $about = $db->loadResult();
        
        return json_decode($about);
    }

    public static function saveCodeEditor($obj, $id)
    {
        $file = JPATH_ROOT. '/templates/gridbox/css/storage/code-editor-'.$id.'.css';
        File::write($file, (string)$obj->css);
        $file = JPATH_ROOT. '/templates/gridbox/js/storage/code-editor-'.$id.'.js';
        File::write($file, (string)$obj->js);
    }

    public static function replace($str)
    {

        $str = mb_strtolower($str, 'utf-8');
        $search = ['?', '!', '.', ',', ':', ';', '*', '(', ')', '{', '}', '***91;',
            '***93;', '%', '#', '№', '@', '$', '^', '-', '+', '/', '\\', '=',
            '|', '"', '\'', 'а', 'б', 'в', 'г', 'д', 'е', 'ё', 'з', 'и', 'й',
            'к', 'л', 'м', 'н', 'о', 'п', 'р', 'с', 'т', 'у', 'ф', 'х', 'ъ',
            'ы', 'э', ' ', 'ж', 'ц', 'ч', 'ш', 'щ', 'ь', 'ю', 'я'];
        $replace = ['-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-',
            '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-', '-',
            'a', 'b', 'v', 'g', 'd', 'e', 'e', 'z', 'i', 'y', 'k', 'l', 'm', 'n',
            'o', 'p', 'r', 's', 't', 'u', 'f', 'h', 'j', 'i', 'e', '-', 'zh', 'ts',
            'ch', 'sh', 'shch', '', 'yu', 'ya'];
        $str = str_replace($search, $replace, $str);
        $str = trim($str);
        $str = preg_replace("/_{2,}/", "-", $str);

        return $str;
    }

    public static function setAppLicenseBalbooa($data)
    {
        $balbooa = self::getGridboxApi('balbooa_activation');
        if (!$balbooa->key) {
            $balbooa->key = self::checkGridboxState();
        }
        $balbooa->key = json_decode($balbooa->key);
        $balbooa->key->data = $data;
        if (empty($data)) {
            unset($balbooa->key->data);
        }
        $balbooa->key = json_encode($balbooa->key);
        $db = Factory::getDbo();
        $db->updateObject('#__gridbox_api', $balbooa, 'id');
    }

    public static function setAppLicense($data)
    {
        $db = Factory::getDbo();
        $balbooa = self::getGridboxApi('balbooa');
        $balbooa->key = json_decode($balbooa->key);
        $balbooa->key->data = $data;
        if (empty($data)) {
            unset($balbooa->key->data);
        }
        $balbooa->key = json_encode($balbooa->key);
        $db->updateObject('#__gridbox_api', $balbooa, 'id');
    }

    public static function getTaxCountries($setObject = null)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_countries')
            ->order('title ASC');
        $db->setQuery($query);
        $countries = $db->loadObjectList();
        if ($setObject) {
            $data = new \stdClass();
            foreach ($countries as $country) {
                $data->{$country->id} = $country;
            }
            $countries = $data;
        }
        foreach ($countries as $country) {
            $query = $db->getQuery(true)
                ->select('*')
                ->from('#__gridbox_country_states')
                ->where('country_id = '.$country->id)
                ->order('title ASC');
            $db->setQuery($query);
            $country->states = $db->loadObjectList();
        }
        
        return $countries;
    }

    public static function checkGridboxState()
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('`key`')
            ->from('#__gridbox_api')
            ->where('service = '.$db->quote('balbooa'));
        $db->setQuery($query);
        $balbooa = self::getGridboxApi('balbooa');

        return $balbooa->key;
    }

    public static function stringURLSafe($string, $language = '')
    {
        if (Factory::getConfig()->get('unicodeslugs') == 1) {
            $output = OutputFilter::stringURLUnicodeSlug($string);
        } else {
            if ($language === '*' || $language === '') {
                $languageParams = ComponentHelper::getParams('com_languages');
                $language = $languageParams->get('site');
            }
            $output = OutputFilter::stringURLSafe($string, $language);
        }

        return $output;
    }

    public static function increment($string)
    {
        if (preg_match('#\((\d+)\)$#', $string, $matches)) {
            $n = (int) $matches[1] + 1;
            $string = preg_replace('#\(\d+\)$#', sprintf('(%d)', $n), $string);
        } else {
            $n = 2;
            $string .= sprintf(' (%d)', $n);
        }

        return $string;
    }

    public static function getAlias(string $alias, string $table, int $item_id = 0, string $cell = 'alias')
    {
        $originAlias = $alias;
        $alias = self::stringURLSafe(trim($alias));
        if (empty($alias)) {
            $alias = $originAlias;
            $alias = self::replace($alias);
            $alias = OutputFilter::stringURLSafe($alias);
        }
        if (empty($alias)) {
            $alias = date('Y-m-d-H-i-s');
        }
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select('id')
            ->from($table)
            ->where($cell.' = ' .$db->quote($alias))
            ->where('id <> ' .$db->quote($item_id));
        $db->setQuery($query);
        $id = $db->loadResult();
        if (!empty($id)) {
            $alias = self::increment($alias);
            $alias = self::getAlias($alias, $table, $item_id, $cell);
        }
        
        return $alias;
    }

    public static function calculateProductTax($id, $price, $cart, $country = true, $region = true, $category = true)
    {
        $obj = null;
        $array = $category ? self::$taxRates->categories : self::$taxRates->empty;
        foreach ($array as $tax) {
            $count = $country ? $tax->country_id == $cart->country : true;
            $cat = $category ? self::checkProductTaxMap($id, $tax->categories) : true;
            $reg = self::getTaxRegion($tax->regions, $cart->region, $region);
            if ($count && $cat && $reg) {
                $rate = !empty($reg->rate) ? $reg->rate : $tax->rate;
                $obj = (object)[
                    'key' => $tax->key,
                    'title' => $tax->title,
                    'rate' => $rate,
                    'amount' => self::$store->tax->mode == 'excl' ? $price * ($rate / 100) : $price - $price / ($rate / 100 + 1)
                ];
                break;
            }
        }
        if (!$obj && $country && $region && $category) {
            $obj = self::calculateProductTax($id, $price, $cart, true, false, true);
        } else if (!$obj && $country && !$region && $category) {
            $obj = self::calculateProductTax($id, $price, $cart, true, true, false);
        } else if (!$obj && $country && $region && !$category) {
            $obj = self::calculateProductTax($id, $price, $cart, true, false, false);
        } else if (!$obj && $country && !$region && !$category) {
            $obj = self::calculateProductTax($id, $price, $cart, false, false, true);
        } else if (!$obj && !$country && !$region && $category) {
            $obj = self::calculateProductTax($id, $price, $cart, false, false, false);
        }

        return $obj;
    }

    public static function getTaxRegion($regions, $region, $flag)
    {
        $result = new \stdClass();
        if (!$flag) {
            return $result;
        }
        foreach ($regions as $value) {
            if ($value->state_id == $region) {
                $result = $value;
                break;
            }
        }

        return $result;
    }

    public static function checkProductTaxMap($id, $categories)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('page_category')
            ->from('#__gridbox_pages')
            ->where('id = '.$id);
        $db->setQuery($query);
        $category = $db->loadResult();
        $flag = self::checkProductCategory($category, $categories);

        return $flag;
    }

    public static function checkProductCategory($id, $array)
    {
        $flag = in_array($id, $array);
        if (!$flag && !empty($id)) {
            $db = Factory::getDbo();
            $query = $db->getQuery(true)
                ->select('parent')
                ->from('#__gridbox_categories')
                ->where('id = '.$id);
            $db->setQuery($query);
            $category = $db->loadResult();
            $flag = self::checkProductCategory($category, $array);
        }

        return $flag;
    }

    public static function getStoreShippingTax($cart, $country = true, $region = true)
    {
        $obj = null;
        foreach (self::$store->tax->rates as $key => $rate) {
            $count = $country ? $rate->country_id == $cart->country : true;
            $reg = self::getTaxRegion($rate->regions, $cart->region, $region);
            if ($rate->shipping && $count && $reg) {
                $obj = new \stdClass();
                $obj->key = $key;
                $obj->title = $rate->title;
                $obj->rate = $rate->rate;
                $obj->amount = $rate->rate / 100;
                break;
            }
        }
        if (!$obj && $country && $region) {
            $obj = self::getStoreShippingTax($cart, true, false);
        } else if (!$obj && $country && !$region) {
            $obj = self::getStoreShippingTax($cart, false, false);
        }

        return $obj;
    }

    public static function getStoreShippingItem($item, $total, $tax, $cart)
    {
        $mode = self::$store->tax->mode;
        $item->params = json_decode($item->options);
        $type = $item->params->type;
        $object = isset($item->params->{$type}) ? $item->params->{$type} : null;
        if ($type == 'free' || $type == 'pickup') {
            $item->price = 0;
        } else if ($type == 'flat') {
            $item->price = $object->price;
        } else if ($type == 'weight-unit') {
            $weight = 0;
            foreach ($cart->products as $product) {
                if (!empty($product->data->weight)) {
                    $weight += $product->data->weight * $product->quantity;
                } else if (!empty($product->dimensions->weight)) {
                    $weight += $product->dimensions->weight * $product->quantity;
                }
                foreach ($product->extra_options->items as $extra) {
                    foreach ($extra->values as $value) {
                        if (!empty($value->weight)) {
                            $weight += $value->weight * $product->quantity;
                        }
                    }
                }
            }
            $item->price = $weight * $object->price;
        } else if ($type == 'product') {
            $item->price = $cart->quantity * $item->params->product->price;
        } else if ($type == 'prices' || $type == 'weight') {
            $range = [];
            $unlimited = null;
            foreach ($object->range as $value) {
                if ($value->rate === '') {
                    $unlimited = $value;
                } else {
                    $value->rate *= 1;
                    $range[] = $value;
                }
            }
            usort($range, function($a, $b){
                if ($a->rate == $b->rate) {
                    return 0;
                }

                return ($a->rate < $b->rate) ? -1 : 1;
            });
            $price = null;
            if ($type == 'weight') {
                $netValue = 0;
                foreach ($cart->products as $product) {
                    if (!empty($product->data->weight)) {
                        $netValue += $product->data->weight * $product->quantity;
                    } else if (!empty($product->dimensions->weight)) {
                        $netValue += $product->dimensions->weight * $product->quantity;
                    }
                    foreach ($product->extra_options->items as $extra) {
                        foreach ($extra->values as $value) {
                            if (!empty($value->weight)) {
                                $netValue += $value->weight * $product->quantity;
                            }
                        }
                    }
                }


            } else {
                $netValue = $cart->total;
            }
            foreach ($range as $value) {
                if ($netValue <= $value->rate) {
                    $price = $value;
                    break;
                }
            }
            if ($price === null && $unlimited) {
                $price = $unlimited;
            }
            if ($price) {
                $item->price = $price->price;
            } else {
                $item->price = 0;
            }
        } else if ($type == 'category') {
            $item->price = 0;
            foreach ($cart->products as $product) {
                if (empty($product->product_id)) {
                    continue;
                }
                $categories = self::getCategoryId($product->product_id);
                $price = null;
                foreach ($item->params->category->range as $range) {
                    foreach ($range->rate as $id) {
                        if (in_array($id, $categories)) {
                            $price = $range->price;
                            break;
                        }
                    }
                    if ($price !== null) {
                        break;
                    }
                }
                if ($price !== null) {
                    $item->price += $price * $product->quantity;
                    continue;
                }
            }
        }
        if ($object && isset($object->enabled) && $object->enabled && $total > $object->free * 1) {
            $item->price = 0;
        }
        $amount = $tax ? $tax->amount : 0;
        $item->tax = $mode == 'excl' ? $item->price * $amount : $item->price - $item->price / ($amount + 1);
        $item->total = $total + $item->price + ($mode == 'excl' ? $item->tax : 0);

        return $item;
    }

    public static function getCategoryId($id)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true);
        $query->select('page_category')
            ->from('#__gridbox_pages')
            ->where('id = '.$id);
        $db->setQuery($query);
        $category = $db->loadResult();
        $array = [$category];
        $array2 = self::getCategoryIdPath($category);
        $result = array_merge($array, $array2);
        
        return $result;
    }

    public static function getCategoryIdPath($id)
    {
        if (isset(self::$cacheData->categories->path)) {
            $categories = self::$cacheData->categories->path;
        } else {
            $db = Factory::getDbo();
            $query = $db->getQuery(true)
                ->select('id, parent')
                ->from('#__gridbox_categories');
            $db->setQuery($query);
            $categories = $db->loadAssocList('id', 'parent');
            self::addCacheData($categories, 'categories', 'path');
        }
        $path = [];
        while (!empty($categories[$id]) && $categories[$id] != 0) {
            $id = $categories[$id];
            $path[] = $id;
        }

        return $path;
    }

    public static function checkPromoCode($promo, $product)
    {
        $valid = false;
        if ($promo->applies_to != '*') {
            $db = Factory::getDbo();
            $query = $db->getQuery(true)
                ->select('*')
                ->from('#__gridbox_store_promo_codes_map')
                ->where('code_id = '.$promo->id)
                ->where('type = '.$db->quote($promo->applies_to));
            $db->setQuery($query);
            $promo->map = $db->loadObjectList();
        }
        if (!empty($product->upgrade_id) && $promo->disable_plan_change == 1) {
            return false;
        }
        if ($promo->applies_to == '*' && $promo->disable_sales == 0) {
            $valid = true;
        } else if ($promo->applies_to == '*' && $promo->disable_sales == 1) {
            $data = self::$storeHelper->getProductData($product->product_id);
            $prices = self::prepareProductPrices($data->product_id, $data->price, $data->sale_price, $product->variation);
            $valid = $prices->sale_price !== '' ? false : true;
        } else if ($promo->applies_to == 'product') {
            foreach ($promo->map as $value) {
                if ($product->product_id == $value->item_id &&
                    $product->variation == $value->variation && $promo->disable_sales == 0) {
                    $valid = true;
                } else if ($product->product_id == $value->item_id && $product->variation == $value->variation
                    && $promo->disable_sales == 1) {
                    $data = self::$storeHelper->getProductData($product->product_id);
                    $prices = self::prepareProductPrices($data->product_id, $data->price, $data->sale_price, $product->variation);
                    $valid = $prices->sale_price !== '' ? false : true;
                }
                if ($valid) {
                    break;
                }
            }
        } else {
            $categories = self::getCategoryId($product->product_id);
            foreach ($promo->map as $value) {
                if (in_array($value->item_id, $categories) && $promo->disable_sales == 0) {
                    $valid = true;
                } else if (in_array($value->item_id, $categories) && $promo->disable_sales == 1) {
                    $data = self::$storeHelper->getProductData($product->product_id);
                    $prices = self::prepareProductPrices($data->product_id, $data->price, $data->sale_price, $product->variation);
                    $valid = $prices->sale_price !== '' ? false : true;
                }
                if ($valid) {
                    break;
                }
            }
        }
        
        return $valid;
    }

    public static function calculateSalePrice($sales, $price, $sale_price, $id, $variation)
    {
        $groups = Factory::getUser()->getAuthorisedViewLevels();

        return self::prepareSalePrice($sales, $price, $sale_price, $id, $variation, $groups);
    }

    public static function prepareSalePrice($sales, $price, $sale_price, $id, $variation, $groups)
    {
        foreach ($sales as $sale) {
            if ($sale_price !== '' || empty($sale->discount) ||
                (!empty($groups) && !in_array($sale->access, $groups))) {
                continue;
            }
            $applies = false;
            if ($sale->applies_to == '*') {
                $applies = true;
            } else if ($sale->applies_to == 'category') {
                $categories = self::getCategoryId($id);
                foreach ($sale->map as $value) {
                    $applies = in_array($value->item_id, $categories);
                    if ($applies) {
                        break;
                    }
                }
            } else if ($sale->applies_to == 'product') {
                foreach ($sale->map as $value) {
                    $applies = $value->item_id == $id && $value->variation == $variation;
                    if ($applies) {
                        break;
                    }
                }
            }
            if (!$applies) {
                continue;
            }
            $sale_price = $price - ($sale->unit == '%' ? $price * ($sale->discount / 100) : $sale->discount);
        }

        return $sale_price;
    }

    public static function prepareProductPrices($id, $price, $sale_price, $variation = '', $qty = 1)
    {
        $price = floatval($price) * $qty;
        $currency = self::$store->currency;
        $sales = self::$storeHelper->sales;
        $prices = new \stdClass();
        $prices->price = $price;
        $prices->regular = self::prepareProductPrice($price, $currency->thousand, $currency->separator, $currency->decimals);
        $prices->sale = '';
        $prices->sale_price = $sale_price;
        $sale_price = self::calculateSalePrice($sales, $price, $sale_price, $id, $variation);
        if ($sale_price !== '') {
            $sale_price = floatval($sale_price) * $qty;
            $prices->sale_price = $sale_price;
            $prices->sale = self::prepareProductPrice($sale_price, $currency->thousand, $currency->separator, $currency->decimals);
        }

        return $prices;
    }

    public static function prepareProductPrice($price, $thousand, $separator, $decimals, $rate = null)
    {
        $price = floatval($price);
        if (!$rate) {
            $rate = self::$store->currency->rate;
        }
        $price = round($price * $rate, $decimals);
        $price = number_format($price, $decimals, $separator, $thousand);

        return $price;
    }

    public static function getGridboxApi(string $service):?object
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_api')
            ->where('service = '.$db->quote($service));
        $db->setQuery($query);
        $obj = $db->loadObject();

        return $obj;
    }

    public static function getSystemParamsByType($type)
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_system_pages')
            ->where('published = 1')
            ->where('language in ('.$db->quote(Factory::getLanguage()->getTag()).','.$db->quote('*').')')
            ->where('type = '.$db->quote($type));
        $db->setQuery($query);
        $obj = $db->loadObject();

        return $obj;
    }
}