<?php
/**
* @package   Gridbox
* @author    Balbooa http://www.balbooa.com/
* @copyright Copyright @ Balbooa
* @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
*/

namespace Balbooa\Component\Gridbox\Site\Helper\Dashboard;

use Joomla\CMS\Factory;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Date\Date;
use Balbooa\Component\Gridbox\Administrator\Helper\GridboxHelper;
use Balbooa\Component\Gridbox\Site\Interface\DashboardInterface;

class SalesHelper implements DashboardInterface
{
    private object $statistic;
    private array $dates;

    public function __construct()
    {
        $this->dates = [];
        $this->statistic = (object)[
            'total' => 0,
            'counts' => [
                'orders' => 0,
                'completed' => 0,
                'refunded' => 0
            ],
            'products' => [],
            'chart' => []
        ];
    }

    public static function make() :static
    {
        return new static();
    }

    private function prepareDates(string $date, string $type):void
    {
        switch ($type) {
            case 'd':
                $this->prepareDayDates($date);
                break;

            case 'w':
                $this->prepareWeekDates($date);
                break;

            case 'm':
                $this->prepareMonthDates($date);
                break;

            case 'y':
                $date = $this->getFirstOrderDate();
                $this->prepareYearDates($date);
                break;

            case 'c':
                $this->prepareCustomDates($date);
                break;
        }
    }

    private function prepareDayDates(string $date):void
    {
        $this->appendDate($date, $date, Date::getInstance($date)->format('M d, Y'));
    }

    private function prepareWeekDates($date):void
    {
        for ($i = 7; $i > 0; $i--) {
            $d = date('Y-m-d', strtotime($date . ' -' . $i . ' days'));
            $this->appendDate($d, $d, Date::getInstance($d)->format('D'));
        }
    }

    private function prepareMonthDates($date):void
    {
        for ($i = 1; $i <= 12; $i++) {
            $d = date('Y-m-d', strtotime($date . '-' . ($i < 10 ? '0' . $i : $i) . '-01'));
            $this->appendDate($d, date('Y-m-t', strtotime($d)), Date::getInstance($d)->format('M'));
        }
    }

    private function prepareYearDates($date):void
    {
        $current = date('Y');
        for ($i = date('Y', strtotime($date)); $i <= $current; $i++) {
            $d = date('Y-m-d', strtotime($i . '-01-01'));
            $this->appendDate($d, date('Y-m-t', strtotime($i)), Date::getInstance($d)->format('Y'));
        }
    }

    private function prepareCustomDates($date):void
    {
        $dates = explode(' - ', $date);
        $label = Date::getInstance($dates[0])->format('M d, Y')
            . ' - ' . Date::getInstance($dates[0])->format('M d, Y');
        $this->appendDate($dates[0], $dates[1], $label);
    }

    private function appendDate(string $start, string $end, string $label):void
    {
        $this->dates[] = (object)[
            'start' => $start,
            'end' => $end,
            'label' => $label
        ];
    }

    public function getFirstOrderDate() :?string
    {
        $db = Factory::getDbo();
        $query = $db->getQuery(true)
            ->select('date')
            ->from('#__gridbox_store_orders')
            ->where('published = 1')
            ->order('date ASC');
        $db->setQuery($query);
        $date = $db->loadResult();

        return $date;
    }

    public function getShopStatistic($date, $type) :object
    {
        $this->prepareDates($date, $type);
        foreach ($this->dates as $date) {
            $this->getStatisticData($date->start, $date->end, $date->label);
        }

        return $this->statistic;
    }
   
    public function getStatisticData($start, $end, $date)
    {
        $db = Factory::getDbo();
        $start = $db->quote($start . ' 00:00:01');
        $end = $db->quote($end . ' 23:59:59');
        $query = $db->getQuery(true)
            ->select('total, tax, status, id')
            ->from('#__gridbox_store_orders')
            ->where('published = 1')
            ->where('date > ' . $start)
            ->where('date < ' . $end);
        $db->setQuery($query);
        $orders = $db->loadObjectList();
        

        $total = 0;
        $pks = [];
        $products = [];
        foreach ($orders as $order) {
            if (isset($this->statistic->counts[$order->status])) {
                $this->statistic->counts[$order->status]++;
            }
            if ($order->status == 'completed') {
                $pks[] = $order->id;
            }
            $this->statistic->counts['orders']++;
        }
        if (!empty($pks)) {
            $str = implode(', ', $pks);
            $products = $this->getStatisticProducts($str);
            foreach ($products as $product) {
                if (count($this->statistic->products) < 10) {
                    $this->statistic->products[] = $product;
                }
                $this->statistic->total += $product->price;
                $total += $product->price;
            }
            foreach ($pks as $pk) {
                $query = $db->getQuery(true)
                    ->select('*')
                    ->from('#__gridbox_store_orders_discount')
                    ->where('order_id = '.$pk);
                $db->setQuery($query);
                $promo = $db->loadObject();
                if ($promo) {
                    $this->statistic->total -= $promo->value;
                    $total -= $promo->value;
                }
            }
        }
        $this->statistic->chart[] = (object)[
            'label' => $date,
            'value' => $total,
        ];
        
    }

    public function getStatisticProducts($orders_id)
    {
        $db = Factory::getDbo();
        $user = Factory::getUser();
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_store_order_products')
            ->where('order_id IN (' . $orders_id . ')');
        $db->setQuery($query);
        $items = $db->loadObjectList();
        $object = new \stdClass();
        foreach ($items as $item) {
            $id = !empty($item->variation) ? $item->variation : $item->product_id;
            $item->quantity = $item->quantity * 1;
            if ($item->sale_price != '') {
                $item->price = $item->sale_price;
            }
            if (!isset($object->{$id})) {
                $object->{$id} = $item;
            } else {
                $object->{$id}->quantity += $item->quantity;
                $object->{$id}->price += $item->price;
            }
        }
        $products = [];
        foreach ($object as $obj) {
            $products[] = $obj;
        }
        uasort($products, function($a, $b){
            if ($a->price == $b->price) {
                return 0;
            }
            return ($a->price < $b->price) ? 1 : -1;
        });
        foreach ($products as $product) {
            $query = $db->getQuery(true)
                ->select('*')
                ->from('#__gridbox_store_order_product_variations')
                ->where('product_id = ' . $product->id);
            $db->setQuery($query);
            $variations = $db->loadObjectList();
            $info = [];
            foreach ($variations as $variation) {
                $info[] = '<span>' . $variation->title . ' ' . $variation->value . '</span>';
            }
            $product->info = implode('/', $info);
            if (!empty($product->image) && !GridboxHelper::isExternal($product->image)) {
                $product->image = Uri::root() . str_replace(' ', '%20', $product->image);
            }
            $product->icon = GridboxHelper::getIcon($product);
            $query = $db->getQuery(true)
                ->select('p.id')
                ->from('#__gridbox_pages AS p')
                ->where('d.product_id = '.$product->product_id)
                ->leftJoin('#__gridbox_store_product_data AS d ON d.product_id = p.id');
            $db->setQuery($query);
            $product->product_id = $db->loadResult();
            $product->link = '';
            if ($product->product_id && $user->authorise('core.edit', 'com_gridbox.page.' . $product->product_id)) {
                $product->link = 'index.php?option=com_gridbox&task=gridbox.edit&id=' . $product->product_id;
            }
        }

        return $products;
    }
}