<?php
/**
* @package   Gridbox
* @author    Balbooa http://www.balbooa.com/
* @copyright Copyright @ Balbooa
* @license   http://www.gnu.org/licenses/gpl.html GNU/GPL
*/

namespace Balbooa\Component\Gridbox\Site\Helper;

use Joomla\CMS\Factory;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\Database\DatabaseInterface;

defined('_JEXEC') or die;

class SeoHelper
{
    private object $item;
    private string $type;
    private object $tags;
    private array $shortCodes;
    private ?object $product;

    public function __construct($item, $type)
    {
        $this->item = $item;
        $this->type = $type;
        $this->shortCodes = $this->getShortCodes();
        $this->tags = new \stdClass();
    }

    public function prepareText($text, $isSchema = false)
    {
        if (empty($text)) {
            return $text;
        }
        foreach ($this->shortCodes as $tag) {
            preg_match_all("/\[$tag]/", $text, $matches);
            if (empty($matches)) {
                continue;
            }
            foreach ($matches[0] as $match) {
                $value = $this->checkTag($tag, $match, $isSchema);
                $text = str_replace($match, $value, $text);
            }
        }

        return $text;
    }

    public function prepareSchema($text)
    {
        return $this->prepareText($text, true);
    }

    protected function checkTag($tag, $match, $isScheme = false)
    {
        if (isset($this->tags->{$tag})) {
            return $this->tags->{$tag};
        }
        $store = GridboxHelper::$store;
        $value = match ($tag) {
            'Field \d+' => $this->getFieldValue($match, $isScheme),
            'Product SKU' => $this->getProductInfo('sku'),
            'Product In Stock' => $this->getProductInfo('stock', $isScheme),
            'Product Price' => $this->getProductInfo('price'),
            'Product Sale Price' => $this->getProductInfo('sale_price'),
            'Product Currency' => GridboxHelper::$store->currency->code ?? '',
            'Page Title', 'Tag Title', 'Author Name' => $this->item->title ?? '',
            'Page Image' => $this->getPageImage($isScheme),
            'Page URL', 'Category Page URL', 'Tag Page URL', 'Author Page URL' => Uri::current(),
            'Site Name' => Factory::getApplication()->get('sitename'),
            'Date Modified' => $this->getDateModified(),
            'Author' => $this->getAuthors(),
            'Page Tags' => $this->getPageTags(),
            'Start Publishing' => HTMLHelper::date($this->item->created, GridboxHelper::$dateFormat),
            'Intro Text' => $this->item->intro_text,
            'Category Title' => $this->type == 'page' ? $this->item->category_title : $this->item->title,
            'Category Image URL', 'Tag Image URL', 'Author Profile Picture URL' => $this->item->image,
            'Category Description', 'Tag Description', 'Author Description' => $this->item->description,
            'Store Name' => $store->general->store_name,
            'Store Legal Business Name' => $store->general->business_name,
            'Store Phone' => $store->general->phone,
            'Store Email' => $store->general->email,
            'Store Address' => $this->getStoreAddress($store),
            default => '',
        };
        if ($tag == 'Field \d+') {
            $tag = $match;
        }
        $this->tags->{$tag} = strip_tags($value);

        return $this->tags->{$tag};
    }

    protected function getPageImage(bool $isScheme): string
    {
        $image = $this->item->intro_image ?? '';
        if ($isScheme && !GridboxHelper::isExternal($image)) {
            $image = Uri::root() . $image;
        }

        return $image;
    }

    protected function getFieldValue(string $match, bool $isScheme): string
    {
        preg_match('/\d+/', $match, $matches);
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $id = $matches[0];
        if (!isset(GridboxHelper::$cacheData->seoFields->{$id})) {
            $query = $db->getQuery(true)
                ->select('f.*')
                ->from('#__gridbox_fields as f')
                ->where('f.id = ' . $id)
                ->select('pf.value')
                ->where('pf.page_id = ' . $this->item->id)
                ->leftJoin('`#__gridbox_page_fields` AS pf ON pf.field_id = f.id');
            $db->setQuery($query);
            $data = $db->loadObject();
            GridboxHelper::addCacheData($data, 'seoFields', $id);
        } else {
            $data = GridboxHelper::$cacheData->seoFields->{$id};
        }

        if (!$data) {
            return '';
        }

        return match ($data->field_type) {
            'radio', 'select' => $this->getFieldRadio($data),
            'checkbox' => $this->getFieldCheckbox($data),
            'date', 'event-date' => $this->getFieldDate($isScheme, $data),
            'time' => $this->getFieldTime($data),
            'url' => $this->getFieldUrl($data),
            'price' => $this->getFieldPrice($data),
            default => $data->value,
        };
    }

    protected function getFieldOptionsTitles(array $selected, object $options): string
    {
        $result = [];
        foreach ($options->items as $option) {
            if (in_array($option->key, $selected, true)) {
                $result[] = $option->title;
            }
        }

        return implode(', ', $result);
    }


    protected function getFieldRadio(object $data): string
    {
        return $this->getFieldOptionsTitles([$data->value], json_decode($data->options));
    }

    protected function getFieldCheckbox(object $data): string
    {
        return $this->getFieldOptionsTitles(json_decode($data->value), json_decode($data->options));
    }

    protected function getFieldTime(object $data): string
    {
        $valueOptions = json_decode($data->value);

        return "$valueOptions->hours:$valueOptions->minutes $valueOptions->format";
    }

    protected function getFieldUrl(object $data): string
    {
        $valueOptions = json_decode($data->value);

        return GridboxHelper::prepareGridboxLinks($valueOptions->link);
    }

    protected function getFieldDate(bool $isScheme, object $data): string
    {
        if (!$isScheme) {
            return GridboxHelper::formatDate($data->value);
        }

        return (new \DateTime($data->value))->format("Y-m-d\TH:i:s\Z");
    }

    protected function getFieldPrice(object $data): string
    {
        $fieldOptions = json_decode($data->options);
        $thousand = $fieldOptions->thousand;
        $separator = $fieldOptions->separator;
        $decimals = $fieldOptions->decimals;

        return GridboxHelper::preparePrice($data->value, $thousand, $separator, $decimals, 1);
    }

    protected function getProductInfo(string $key, bool $isScheme = false): string
    {
        if (empty($this->product)) {
            $this->product = GridboxHelper::$storeHelper->getProductData($this->item->id);
        }
        if ($isScheme && $key == 'stock') {
            return sprintf(
                'https://schema.org/%s',
                $this->product->stock === '0' ? 'OutOfStock' : 'InStock'
            );
        }

        return $this->product->{$key} ?? '';
    }

    protected function getDateModified(): string
    {
        if (empty($this->item->saved_time)) {
            return '';
        }
        $array = explode('-', $this->item->saved_time);
        $time = count($array) == 6 ? $array[0].'-'.$array[1].'-'.$array[2].' '.$array[3].':'.$array[4].':'.$array[5] : 'now';

        return HTMLHelper::date($time, GridboxHelper::$dateFormat);
    }

    protected function getPageTags(): string
    {
        $groups = implode(',', Factory::getApplication()->getIdentity()->getAuthorisedViewLevels());
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $db->getQuery(true)
            ->select('m.tag_id as id')
            ->from('#__gridbox_tags_map AS m')
            ->where('m.page_id = ' . $this->item->id)
            ->select('t.title')
            ->leftJoin('`#__gridbox_tags` AS t ON m.tag_id = t.id')
            ->order('t.hits desc')
            ->where('t.published = 1')
            ->where('t.language in ('.$db->quote(Factory::getApplication()->getLanguage()->getTag()).','.$db->quote('*').')')
            ->where('t.access in ('.$groups.')')
            ->leftJoin('`#__gridbox_pages` AS p ON m.page_id = p.id');
        $db->setQuery($query);
        $tags = $db->loadObjectList();

        return implode(', ', array_column($tags, 'title'));
    }

    protected function getAuthors(): string
    {
        $array = [];
        foreach ($this->item->authors as $author) {
            $array[] = $author->title;
        }

        return implode(', ', $array);
    }

    protected function getStoreAddress(object $store): string
    {
        $address = [];
        if (!empty($store->general->country)) {
            $address[] = $store->general->country;
        }
        if (!empty($store->general->region)) {
            $address[] = $store->general->region;
        }
        if (!empty($store->general->city)) {
            $address[] = $store->general->city;
        }
        if (!empty($store->general->street)) {
            $address[] = $store->general->street;
        }
        if (!empty($store->general->zip_code)) {
            $address[] = $store->general->zip_code;
        }

        return implode(', ', $address);
    }

    protected function getShortCodes(): array
    {
        $array = [];
        if ($this->type == 'page') {
            $array = [
                'Field \d+', 'Store Address', 'Store Email', 'Store Phone',
                'Store Legal Business Name', 'Store Name', 'Product In Stock',
                'Product Currency', 'Product Sale Price', 'Product Price',
                'Product SKU', 'Intro Text', 'Start Publishing', 'Author',
                'Page Tags', 'Category Title', 'Page URL', 'Page Image',
                'Page Title', 'Site Name', 'Date Modified'
            ];
        } else if ($this->type == 'category') {
            $array = [
                'Site Name', 'Category Title', 'Category Image URL',
                'Category Page URL', 'Category Description'
            ];
        } else if ($this->type == 'tag') {
            $array = [
                'Site Name', 'Tag Title', 'Tag Image URL', 'Tag Page URL',
                'Tag Description'
            ];
        } else if ($this->type == 'author') {
            $array = [
                'Site Name', 'Author Name', 'Author Profile Picture URL',
                'Author Page URL', 'Author Description'
            ];
        }
        return $array;
    }

    public function getGlobal(): object
    {
        $db = Factory::getContainer()->get(DatabaseInterface::class);
        $id = $this->type == 'page' || $this->type == 'category' ? $this->item->app_id : 0;
        $query = $db->getQuery(true)
            ->select('*')
            ->from('#__gridbox_seo_defaults')
            ->where('item_id = '.$id)
            ->where('item_type = '.$db->quote($this->type));
        $db->setQuery($query);
        $obj = $db->loadObject();
        $file = JPATH_ROOT . '/administrator/components/com_gridbox/tmpl/layouts/store-options/product-schema.html';
        if (empty($obj)) {
            $schema = $this->type == 'page' && isset($this->item->app_type) && $this->item->app_type == 'products' ? FileHelper::readFile($file) : '';
            $seo = [
                'id' => '0',
                'item_id' => $id,
                'item_type' => $this->type,
                'meta_title' => '',
                'meta_description' => '',
                'share_image' => '',
                'share_title' => '',
                'share_description' => '',
                'sitemap_include' => '1',
                'changefreq' => 'monthly',
                'priority' => '0.5',
                'schema_markup' => $schema
            ];
            $obj = (object) $seo;
        }

        return $obj;
    }
}